/**
(c) 2020 Contecon Software GmbH Author E. Schreiner

Imagga Addon

Doku:
https://imagga.com/

**/

import de.contecon.picapport.IPhotoMetaData;
import de.contecon.picapport.groovy.IAddonContext;
import de.contecon.picapport.groovy.IAddonExecutionContext;
import de.contecon.picapport.groovy.IAddonFileToProcess;
import de.contecon.picapport.groovy.PhotoFileProcessor;


import org.json.JSONArray;
import org.json.JSONObject;

class Imagga extends PhotoFileProcessor {
  
def ADDON_TYPE_ID = "imagga.com";
  
def en_Title ='Imagga - Automatic photo tagging';
def de_Title ='Imagga - Automatisches taggen von Fotos';
    
def en_Options=['Write only when empty',  'Always overwrite',    'no Changes (just test)',        'Remove Imagga metadata',     'Show Imagga metadata of photos'];
def de_Options=['Schreibe nur wenn leer', 'Immer überschreiben', 'keine Änderungen (nur testen)', 'Imagga Metadaten entfernen', 'Imagga Daten der Fotos anzeigen'];
  
public Map init(IAddonContext addonContext) {
	addonContext.getLogger().logMessage(" Addon loaded. Autor: E. Schreiner (c)2020 Contecon Software GmbH" );

	def meta =  [
                version: '1.0.0', 
				functions: [
						   f1: [
							   name:       en_Title,
							   desc:       'This Addon needs a valid open API-Key\nfrom https://imagga.com/',
							   permission: 'pap:editmeta:photo',
							   
							   parameter: [
                                          mode: [
                                              type:    'select',
                                              label:   'Overwrite',
                                              options: en_Options,
                                              value:   addonContext.getConfigParAsString('mode', '0')
                                              ],
                                          probability: [
                                                type: 'range',
                                                label: 'Probability in percent' ,
                                                value: addonContext.getConfigParAsString('probability', '50'),
                                                min:   '1',
                                                max:   '100',
                                                ],
                                          language: [
                                                type:  'text',
                                                label: 'language',
                                                value:  addonContext.getConfigParAsString('language', System.getProperty("user.language")),
                                                permission: 'pap:admin:addon:config'
                                                ],
                                          apikey: [
                                                type:       'text',
                                                label:      'API-Key',
                                                placeholder:'API-Key from www.imagga.com',
                                                value:      addonContext.getConfigParAsString('apikey', ''),
                                                permission: 'pap:admin:addon:config'
                                                ],
                                          updateDefaults: [
                                                type:       'checkbox',
                                                label:      'Save current parameter values as defaults',
                                                value:      false,
                                                permission: 'pap:admin:addon:config'
                                                ],                                                
                                          analyseResult: [
                                              type: 'checkbox',
                                              label: 'analyse result',
                                              value:  addonContext.getConfigParAsBoolean('analyseResult', false)
                                              ]
										  ]											  
							   ]								   
						   ],
				i18n: [
					  'de.f1.name':                       de_Title,
					  'de.f1.desc':                       'Dieses Addon benötigt einen gültigen API-Key\nvon https://www.imagga.com',
                      'de.f1.mode.label':                 'Überschreiben',
                      'de.f1.mode.options':               de_Options,
					  'de.f1.apikey.label':               'API-Key',
                      'de.f1.probability.label':          'Wahrscheinlichkeit in Prozent',
                      'de.f1.language.label':             'Sprache',
                      'de.f1.apikey.placeholder':         'API-Key von www.imagga.com',
                      'de.f1.updateDefaults.label':       'Aktuelle Parameter als Vorgabe speichern',
					  'de.f1.analyseResult.label':        'Analysiere Ergebnis',
					  ],						
				]
	}

/**
 * Save current parameter values from GUI in config.json in addon directory
 *   
 * author Eric 22.07.2020
 * @param addonContext
 * @param aec
 */
private void handleDefaultParameterUpdate(IAddonContext addonContext, IAddonExecutionContext aec) {
  addonContext.putConfigPar("mode",                 aec.mode);
  addonContext.putConfigPar("probability",          aec.probability);
  addonContext.putConfigPar("language",             aec.language);
  addonContext.putConfigPar("apikey",               aec.apikey);
  addonContext.putConfigPar("analyseResult",        aec.analyseResult);
  addonContext.updateConfigFile();
  }
  
   
/*
 * This method is called one Time befor processing of each photo starts
 * use IAddonExecutionContext aec as a map to store values over the livetime of this procedure 
 */
public void start(IAddonContext addonContext, IAddonExecutionContext aec) {
    aec.setShowResults(true); 
    aec.currentPhoto=0; 
    aec.statisticAdded=0;
    aec.statisticReplaced=0;
    aec.statisticSkipped=0;
    aec.statisticRemoved=0;
    aec.statisticQuery=0;

    def titleMap;
                  
    if(aec.updateDefaults) {
      handleDefaultParameterUpdate(addonContext, aec);
      titleMap =["Run-mode":addonContext.i18n("New parameter settings saved.",  ["de":"Neue Parametervorgaben gespeichert."])
                ];
      aec.signalTermination();
      } else {
      titleMap =["Run-mode":addonContext.i18n(en_Options[aec.mode as Integer],  ["de":de_Options[aec.mode as Integer]]),
                (addonContext.i18n("Language",  ["de":"Sprache"])):aec.language
                ];
      }
    aec.getPhotoFileProcessorResultGenerator().addGroupData(addonContext.i18n(en_Title,  ["de":de_Title] ), titleMap);
    }
  
    
/**
 * This method is called for each selected photo
 */
public void processPhotoFile(IAddonContext addonContext, IAddonExecutionContext aec, IAddonFileToProcess fileToProcess) {
    aec.currentPhoto++;
  
    def mode = aec.mode as Integer;
    def typAlreadyExists = fileToProcess.hasAddonMetadata(ADDON_TYPE_ID) as boolean;
    def json = null as JSONObject;
    
    File originalFile = fileToProcess.getOriginalFile();
    def baseAttrs=[:] as LinkedHashMap; // Content of this map will always be visible below the image (Filename etc.)
        baseAttrs['Name']         = originalFile.getName();
        baseAttrs['Path']         = originalFile.getParent();
        baseAttrs['Length']       = originalFile.length();
        baseAttrs['Last Modified']= new Date(originalFile.lastModified());

    if(mode == 3 || mode == 4) { // Remove or Show "osm.org" Metadata call to osm.org is not required
      if(typAlreadyExists) {
        // remove it
        if(mode == 3) { // remove
          fileToProcess.removeAddonMetadata(addonContext, ADDON_TYPE_ID);
          baseAttrs[ADDON_TYPE_ID+' Metadata']= "[[color:#8bc34a;]]Removed";
          aec.statisticRemoved++;
          } else {
          json = fileToProcess.getAddonMetadataAsJSON(ADDON_TYPE_ID);
          baseAttrs[ADDON_TYPE_ID+' Metadata from Photo']= "[[color:#8bc34a;]]Exists";
          aec.statisticQuery++;
          }
        } else {
        // nothing to do
        baseAttrs[ADDON_TYPE_ID+' Metadata']= "[[color:#DD130E;]]"+addonContext.i18n("Imagga metadata not found in photo",  ["de":"Keine Imagga Metadaten im Foto gefunden"] );
        aec.statisticSkipped++;
        }
      if(aec.analyseResult || mode ==4) {
         aec.getPhotoFileProcessorResultGenerator().addGroupData("Result", baseAttrs);
         if(null != json) {
           aec.getPhotoFileProcessorResultGenerator().addGroupData("json", json);
           }
         }
      return;
      }
  
    IPhotoMetaData photoMetadata=fileToProcess.getPhotoMetadata();
    
    def saveJSON = (1 == mode) as boolean; // mode 1 = Always overwrite
    
    if(0 == mode) {
      if(typAlreadyExists) {
        def alreadyExists=addonContext.i18n("Imagga-Data already loaded", ["de":"Imagga Daten bereits geladen"]);
        baseAttrs[ADDON_TYPE_ID+' Metadata']= "[[color:#8bc34a;]]"+alreadyExists;
        if(aec.analyseResult) {
           aec.getPhotoFileProcessorResultGenerator().addGroupData(alreadyExists, baseAttrs);
           }
        aec.statisticSkipped++;
        return;
        } else {
        saveJSON = true; // osm.org does not exist load and save json from osm
        }
      }
    
    
    def params = [language: aec.language];
    
    def url =new URL("https://api.imagga.com/v2/tags?"+(params.collect { k,v -> "$k=$v" }.join('&')));
    addonContext.getLogger().logDebugMessage(" "+url);
	
    String crlf = "\r\n";
    String twoHyphens = "--";
    String boundary =  "Image Upload";
	
    def connection=url.openConnection();
        connection.setDoOutput(true);
        connection.setDoInput(true);
		connection.setUseCaches(false);
        connection.requestMethod = 'POST';
        connection.setRequestProperty("Authorization","Basic "+aec.apikey);
        connection.setRequestProperty("Accept",       "application/json");
        connection.setRequestProperty("User-Agent",   "PicApportGroovy/1.0");
		connection.setRequestProperty("Connection",   "Keep-Alive");
        connection.setRequestProperty("Cache-Control","no-cache");
		connection.setRequestProperty("Content-Type", "multipart/form-data;boundary=" + boundary);
println("FIXME "+connection);									 		
        connection.getOutputStream().withCloseable { os ->
		                             DataOutputStream request = new DataOutputStream(os);
		                             request.writeBytes(twoHyphens + boundary + crlf);
									 request.writeBytes("Content-Disposition: form-data; name=\"image\";filename=\"" + originalFile.getName() + "\"" + crlf);
                                     request.writeBytes(crlf);
									 request.flush();
                                     fileToProcess.writeScaledJpgWithoutMetadata(os, 1200, 1200, 0.7f);
                                     }
          
    def successful = connection.responseCode == 200;
    
    if(aec.analyseResult || !successful) {
      def color = successful ? "" : "[[color:#DD130E;]]";
      baseAttrs["Request"]=         color + url;
      baseAttrs["Api-Key"]=         color + aec.apikey;
      baseAttrs["Response Code"]=   color + connection.responseCode;
      baseAttrs["Response Message"]=color + connection.responseMessage;
      baseAttrs["Content Type"]=    color + connection.contentType;
      baseAttrs["Content Encoding"]=color + connection.contentEncoding;
      baseAttrs["Content Length"]=  color + connection.contentLength;
      aec.getPhotoFileProcessorResultGenerator().addGroupData("Result", baseAttrs);
      }
      
    if(successful) {
      StringBuilder responseText = new StringBuilder();
      new BufferedReader(new InputStreamReader(connection.getInputStream(), "utf-8")).withCloseable { br ->
          String responseLine = null;
          while((responseLine = br.readLine()) != null) {
               responseText.append(responseLine.trim());
               }
          }
      def jsonWeb=new JSONObject(responseText.toString());
      
      if(aec.analyseResult) {
         aec.getPhotoFileProcessorResultGenerator().addGroupData("Web Service Result", jsonWeb.toString(2));
         }
         
      def jsonFiltered=filterResult(addonContext, aec, jsonWeb);
            
      if(aec.analyseResult) {
        def info=(saveJSON ? ADDON_TYPE_ID : "FILE NOT UPDATED");
        aec.getPhotoFileProcessorResultGenerator().addGroupData("Filtered Result (${info})", jsonFiltered.toString(2));
        }
         
      if(saveJSON) {
        fileToProcess.addAddonMetadata(addonContext, ADDON_TYPE_ID, jsonFiltered); 
        aec.notifySidebarUpdateRequired(); // If the metadata-sidebar is open it should update it's content after execution of the addon
        if(typAlreadyExists) {
          aec.statisticReplaced++;
          } else {
          aec.statisticAdded++;
          }
        } else {
        aec.statisticSkipped++;
        }  
      }  	  
    }
    
private JSONObject filterResult(IAddonContext addonContext, IAddonExecutionContext aec, JSONObject webResult) {
    def filteredResult = new JSONObject();
    def tags      = new JSONArray();
    def landmarks = new JSONArray();
    def probability = Double.parseDouble(aec.probability) / 100;

    filterTagsFromArray(webResult.getJSONObject("result").getJSONArray("tags"), tags, probability,  aec.language);
    
    tags=reorgTags(tags);
  
    filteredResult.put("tags",     tags)
    
    return filteredResult;
    }

private void filterTagsFromArray(arrayFrom, arrayTo, probability, language) {
    for(int i = 0; i < arrayFrom.length(); i++) {
       def tag = arrayFrom.getJSONObject(i);
       def ranking = tag.getDouble("confidence");
	   def probabilityTag=ranking/100;
       if(probability <= probabilityTag) {
         // OK copy result
         arrayTo.put(new JSONObject().put("tag", tag.getJSONObject("tag").optString(language)).put("probability", probabilityTag));
         }       
      }
    }  
    
private JSONArray reorgTags(tags) {
   for(int i = 0; i < tags.length(); i++) {
      def currentI=tags.getJSONObject(i);
      for(int j = 0; j < tags.length(); j++) {
         if(i != j) {
           def currentJ=tags.getJSONObject(j);
           if(currentI.getString("tag").length() <= currentJ.getString("tag").length()) {
             if(currentJ.getString("tag").startsWith(currentI.getString("tag"))) {
               currentI.put("removed", true);
               }
             }
           }   
         }
      } 
   def tagsReturn = new JSONArray() as JSONArray;
   for(int i = 0; i < tags.length(); i++) {
      if(!tags.getJSONObject(i).has("removed")) {
        tagsReturn.put(tags.getJSONObject(i));
        }  
      }
   return tagsReturn;  
   }
 
    
/**
 * called after the last photo has been processed use uded to display
 * a summary board with a navigation button      
 */
public void stop(IAddonContext addonContext, IAddonExecutionContext aec) {
    // Print out statistic what we have done
    aec.getPhotoFileProcessorResultGenerator()
       .addGroupData(addonContext.i18n("PicApport Imagga result",       ["de":"PicApport Imagga Ergebnis"] ), 
                   [(addonContext.i18n("Total processed",               ["de":"Anzahl verarbeitet"])):                 aec.currentPhoto,
                    (addonContext.i18n("Imagga Metadata added",         ["de":"Imagga Daten hinzugefügt"])):           aec.statisticAdded,
                    (addonContext.i18n("Imagga Metadata replaced",      ["de":"Imagga Daten ersetzt"])):               aec.statisticReplaced,
                    (addonContext.i18n("Nothing to do",                 ["de":"Nichts zu tun"])):                      aec.statisticSkipped,
                    (addonContext.i18n("Imagga metadata removed",       ["de":"Imagga Metadata entfernt"])):           aec.statisticRemoved,          
                    (addonContext.i18n("Imagga metadata from photo",    ["de":"Imagga Metadata aus Foto angezeigt"])): aec.statisticQuery, 
                    (addonContext.i18n("Problems", ["de":"Probleme"])): (aec.currentPhoto
                                                                        -aec.statisticAdded
                                                                        -aec.statisticReplaced
                                                                        -aec.statisticSkipped
                                                                        -aec.statisticRemoved
                                                                        -aec.statisticQuery)
                   ]);
    }
			
}
