/**
(c) 2020 Contecon Software GmbH Author E. Schreiner

Openroute Service Create a Route for selected photos. 
Make sure that photos are sorted by date ascending.
This addon will generate a GEOJSON-File to connect all selected
photos to a route.

Doku:
https://openrouteservice.org
https://openrouteservice.org/dev/#/api-docs/v2/directions/{profile}/geojson/post    

**/

import de.contecon.picapport.IPhotoMetaData;
import de.contecon.picapport.groovy.IAddonContext;
import de.contecon.picapport.groovy.IAddonExecutionContext;
import de.contecon.picapport.groovy.IAddonFileToProcess;
import de.contecon.picapport.groovy.PhotoFileProcessor;

import java.nio.file.Files;
import java.nio.file.attribute.FileTime;

import org.json.JSONArray;
import org.json.JSONObject;

class OpenroutGeoJSONRoute extends PhotoFileProcessor {
  
def en_Title ='PicApport GEOJSON route generator';
def de_Title ='PicApport GEOJSON Reiseroute Generator';
  
public Map init(IAddonContext addonContext) {
	addonContext.getLogger().logMessage(" Addon loaded. Autor: E. Schreiner (c)2020 Contecon Software GmbH" );

	def meta =  [
                version:'1.0.0',
				functions: [
						   f1: [
							   name:       'Create GeoJSON route for selected photos',
							   desc:       'This Addon needs a valid open API-Key\nfrom https://openrouteservice.org',
							   permission: 'pap:access:metadata',
							   
							   parameter: [
										  filename: [
											  type:  'text',
											  label: 'Optional filename',
                                              value: addonContext.getConfigParAsString('filename', ''), 
											  ],
                                          minDistMeter: [
                                              type: 'range',
                                              label: 'Minimum distance in meters (linear distance in meter)' ,
                                              value: addonContext.getConfigParAsString('minDistMeter', '1000'),
                                              min:   '1',
                                              max:   '10000',
                                              ],
										  routetype: [
											  type:    'select',
											  label:   'Route type',
											  options: ['Car', 'Cycling', 'Cycling mountain', 'Walking', 'Hiking'],
                                              value:   addonContext.getConfigParAsString('mode', '0')
											  ],	  
									      analyseRequest: [
											  type: 'checkbox',
											  label: 'analyse request',
                                              value: addonContext.getConfigParAsBoolean('analyseRequest', false)
											  ],	
                                          apikey: [
                                              type:  'text',
                                              label: 'API-Key',
                                              placeholder:'API-Key from openrouteservice.org',
                                              value:      addonContext.getConfigParAsString('apikey', ''),
                                              permission: 'pap:admin:addon:config'
                                              ],
                                          updateDefaults: [
                                              type:       'checkbox',
                                              label:      'Save current parameter values as defaults',
                                              value:      false,
                                              permission: 'pap:admin:addon:config'
                                              ],
                                          analyseResult: [
                                              type: 'checkbox',
                                              label: 'analyse result',
                                              value: addonContext.getConfigParAsBoolean('analyseResult', false)
                                              ]
										  ]											  
							   ]								   
						   ],
				i18n: [
					  'de.f1.name':                 'Erzeuge GeoJSON Route für selektierte Fotos',
					  'de.f1.desc':                 'Dieses Addon benötigt einen gültigen API-Key\nvon https://openrouteservice.org',
					  'de.f1.filename.label':       'Optionaler Dateiname', 
                      'de.f1.minDistMeter.label':   'Mindestabstand Waypoints (Luftlinie in Meter)',
					  'de.f1.routetype.options':   ['PKW', 'Fahrrad', 'Fahrrad Mountainbike', 'Laufen', 'Wandern'],
                      'de.f1.apikey.label':         'API-Key',
                      'de.f1.apikey.placeholder':   'API-Key von openrouteservice.org',
                      'de.f1.updateDefaults.label': 'Aktuelle Parameter als Vorgabe speichern',
                      'de.f1.analyseRequest.label': 'Analysiere Anfrage',
					  'de.f1.analyseResult.label':  'Analysiere Ergebnis',
                      
                         'fileCreated':            'File created',
                      'de.fileCreated':            'Erzeugte Datei',
                      
                         'pathToFile':             'Path to file',
                      'de.pathToFile':             'In Verzeichnis',
                      
                         'msgCoorninates':         'At least 2 valid GEO-coordinates required',
                      'de.msgCoorninates':         'Mindestens 2 Fotos mit gültigen GEO-Koordinaten werden benötigt.',   
                      
                         'actionShowDir':          'Show folder',
                      'de.actionShowDir':          'Zeige Verzeichnis'
					  ],						
				]
	}

  /**
   * Save current parameter values from GUI in config.json in addon directory
   *
   * author Eric 22.07.2020
   * @param addonContext
   * @param aec
   */
  private void handleDefaultParameterUpdate(IAddonContext addonContext, IAddonExecutionContext aec) {
    addonContext.putConfigPar("filename",      aec.filename);
    addonContext.putConfigPar("minDistMeter",  aec.minDistMeter);
    addonContext.putConfigPar("routetype",     aec.routetype);
    addonContext.putConfigPar("analyseRequest",aec.analyseRequest);
    addonContext.putConfigPar("apikey",        aec.apikey);
    addonContext.putConfigPar("analyseResult", aec.analyseResult);
    addonContext.updateConfigFile();
    }
  
 
/*
 * This method is called one Time before processing of each photo starts
 * use IAddonExecutionContext aec as a map to store values over the lifetime of this procedure 
 */
public void start(IAddonContext addonContext, IAddonExecutionContext aec) {
    aec.enableWebServiceCall=true;//FIXME TRUE IN PRODUCTION // Set to false for testing purposes to disable the call to the Webservice
    aec.setShowResults(true);     // Indicate that this procedure shows a result to the user
    aec.errorColor="";            // Will be set to red when there is an error returned from the webservice     
    
	aec.roadProfile=["driving-car","cycling-regular","cycling-mountain","foot-walking","foot-hiking"];
    aec.dateForJSONfile        = Long.MAX_VALUE;                                      // Used to find the oldest date of the selected photos 
    aec.minimumDistanceInMeters= Integer.parseInt(aec.minDistMeter);                  // Convert entered value from String to int
    aec.coordinates            = new JSONArray();                                     // Array to store lat/lon values 
	aec.radiuses               = new JSONArray();                                     // Array to store radiuses
    aec.requestPOSTjson        = new JSONObject().put("coordinates",aec.coordinates)  // JSON Object that is posted to Web-Service
	                                             .put("radiuses",   aec.radiuses)     // Array to store radiuses
                                                 .put("instructions", false);         // No instructions required  
    aec.totalwaypoints         = 0;                                                   // Count all photos with GEO-coordinates
    aec.summary                = [:] as LinkedHashMap;                                // Map for summary infos
    
    
    def titleMap;
    
    if(aec.updateDefaults) {
      handleDefaultParameterUpdate(addonContext, aec);
      titleMap =["Run-mode":addonContext.i18n("New parameter settings saved.",  ["de":"Neue Parametervorgaben gespeichert."])
                ];
      aec.signalTermination();
      } else {
      titleMap =[(addonContext.i18n("Number of selected photos",  ["de":"Anzahl selektierter Fotos"])):aec.getNumFilesToProcess(),
                ];
      }
    // report header
    aec.getPhotoFileProcessorResultGenerator().addGroupData(addonContext.i18n(en_Title,  ["de":de_Title] ), titleMap);
    }
  
    
/**
 * This method is called for each selected photo
 *   - collects all lat/lon values of the selected photos regarding a minimum distance in meters provided by user
 *   - finds out the oldest date of the selected photos
 *   - if the last selected photo has been processed the web-service is called with all lat/lon coordinates+
 *   - the GeoJSON data returned is stored in a file in the directory of the first selected photo
 *   - the filedate of the geojson file is set to the oldest date fond - one second
 *   - result is displayed to the user with a button to navigate to the directory where the new GeoJSON file has been created      
 */
public void processPhotoFile(IAddonContext addonContext, IAddonExecutionContext aec, IAddonFileToProcess fileToProcess) {
    IPhotoMetaData photoMetadata=fileToProcess.getPhotoMetadata();
    if(photoMetadata.hasGeoCoordinates()) {
      aec.totalwaypoints++;
      def addPoint=true;
      if(aec.coordinates.length() > 0) {
        //calculate distance to last Waypoint
        def distInMeters=distanceInMeters(aec.coordinates.get(aec.coordinates.length()-1).get(0),
                                          aec.coordinates.get(aec.coordinates.length()-1).get(1),
                                          photoMetadata.getLongitude(),
                                          photoMetadata.getLatitude()
                                          );

        if(distInMeters < aec.minimumDistanceInMeters) {
           addPoint=false; // do not add point if distance to last Point is < entered value
           }
        } else {
        // Take the Directory of the first photo as target directory for GeoJSON file
        def path=fileToProcess.getOriginalFile().getParent();
        if(aec.filename.trim().length() == 0) {
          aec.filename=new File(fileToProcess.getOriginalFile().getParent()).getName()+".paRouteGen.geojson";
          }   
        aec.geojsonFile=new File(path+File.separator+aec.filename)  
        }
      if(addPoint) {   
        aec.coordinates.put(new JSONArray().put(photoMetadata.getLongitude()).put(photoMetadata.getLatitude()));
		aec.radiuses.put(-1); // No Limit to search for next road
        }        
      } 
    
    aec.dateForJSONfile=Math.min(aec.dateForJSONfile, photoMetadata.getCreationDate().getTime())-1000; // -1 second because the created JSON File should always be the first file in the directory    
 
    if(aec.isLastFileToProcess()) {
      aec.summary[addonContext.i18n("fileCreated")] = aec.filename;
      aec.summary[addonContext.i18n("pathToFile")]   = fileToProcess.getOriginalFile().getParent();
      if(aec.analyseResult) {
        aec.summary["Minimum distance in meter"]=         aec.minimumDistanceInMeters;
        aec.summary["Total number waypoints"]=            aec.totalwaypoints;
        aec.summary["Total number waypoints removed"]=   (aec.totalwaypoints-aec.coordinates.length());
        aec.summary["Total number waypoints for request"]=aec.coordinates.length();
        }
      
      if(aec.coordinates.length() < 2) {
        throw new RuntimeException(addonContext.i18n('msgCoorninates'));
        }
        		
      if(aec.enableWebServiceCall) { 
        // We call the webservice for theroute
        // see: https://openrouteservice.org/dev/#/api-docs/v2/directions/{profile}/geojson/post
        def url=new URL("https://api.openrouteservice.org/v2/directions/"+aec.roadProfile[aec.routetype as Integer]+"/geojson");
        def connection=url.openConnection();
            connection.setDoOutput(true);
            connection.setDoInput(true);
            connection.requestMethod = 'POST';
            connection.setRequestProperty("Authorization", aec.apikey);
            connection.setRequestProperty("Accept", "application/json, application/geo+json, application/gpx+xml, img/png; charset=utf-8");
            connection.setRequestProperty("Content-Type", "application/json; utf-8");
            connection.setRequestProperty("User-Agent", "PicApportGroovy/1.0");
  
        connection.getOutputStream().withCloseable { os ->
                  byte[] input = aec.requestPOSTjson.toString().getBytes("utf-8");
                  os.write(input, 0, input.length);
                  }
        def successful = connection.responseCode == 200;
                  
        if(aec.analyseResult || !successful) {
           def color = successful ? "" : "[[color:#DD130E;]]";
           aec.summary["Request"]=         color + url;
           aec.summary["Api-Key"]=         color + aec.apikey;
           aec.summary["Response Code"]=   color + connection.responseCode;
           aec.summary["Response Message"]=color + connection.responseMessage;
           aec.summary["Content Type"]=    color + connection.contentType;
           aec.summary["Content Encoding"]=color + connection.contentEncoding;
           aec.summary["Content Length"]=  color + connection.contentLength;
           }
        
        StringBuilder responseText = new StringBuilder();

        new BufferedReader(new InputStreamReader(connection.getInputStream(), "utf-8")).withCloseable { br ->
            String responseLine = null;
            while((responseLine = br.readLine()) != null) {
                 responseText.append(responseLine.trim());
                 }
            }  
        def geoJSON=responseText.toString(); 
		if(successful) {   
          fileToProcess.lockExtraFile(aec.geojsonFile); // It's a good idea to lock all additional files to avoid conflicts with other users or the crawler
                                                        // Locks will automatically be removed when processPhotoFile returns   
          aec.geojsonFile.write(new JSONObject(geoJSON).toString(), "UTF-8");
          Files.setAttribute(aec.geojsonFile.toPath(), "basic:creationTime", FileTime.fromMillis(aec.dateForJSONfile)); // Make sure that the created JSON File is first in directory
          } else {
		  aec.summary["Response Text"]= geoJSON;
		  }
      } else {
        aec.summary["DEMO/TEST MODE"]= "WEBSERVICE HAS NOT BEEN CALLED";
        }
            
      addonContext.rescanDirectoryAfterUpdate(aec.geojsonFile.getParentFile());
      
      aec.summary[addonContext.i18n('actionShowDir')] = addonContext.createAction(aec.geojsonFile.getParent() ,aec.geojsonFile);
      }
    }

    
/**
 * called after the last photo has been processed use uded to display
 * a summary board with a navigation button      
 */
public void stop(IAddonContext addonContext, IAddonExecutionContext aec) {
    if(aec.analyseRequest) {
	   aec.getPhotoFileProcessorResultGenerator().addGroupData("POST Request", aec.requestPOSTjson.toString(2));
	   }
    aec.getPhotoFileProcessorResultGenerator().addGroupData("Summary / Result", aec.summary);
    }

    
/* Simple helper Method to calculate 
 * the distance of two geografical points     
 * I found this on: https://stackoverflow.com/questions/3694380/calculating-distance-between-two-points-using-latitude-longitude   
 */
private static final double r2d = 180.0D / 3.141592653589793D;
private static final double d2r = 3.141592653589793D / 180.0D;
private static final double d2km = 111189.57696D * r2d;
private double distanceInMeters(double lt1, double ln1, double lt2, double ln2) {
    final double x = lt1 * d2r;
    final double y = lt2 * d2r;
    return Math.acos( Math.sin(x) * Math.sin(y) + Math.cos(x) * Math.cos(y) * Math.cos(d2r * (ln1 - ln2))) * d2km;
    }    
			
}
